/////////////////////////////////////////////////////////////////////////
// Muscore 4 plugin
// Add accordion button symbols as lyrics
// Paul Anderson <paul.musescore@nine3.org>
/////////////////////////////////////////////////////////////////////////

import MuseScore 3.0
import QtQuick
import QtQuick.Controls
import QtQuick.Layouts
import QtQuick.Window

import "CustomInstruments.js" as CI
import "CustomTranslations.js" as CT
import "AccordionButtons.js" as AB

MuseScore {

  /////////////////////////////////////////////////////////////////////////
  // properties
  /////////////////////////////////////////////////////////////////////////

  version: "2.3"
  pluginType: "dialog"
  requiresScore: true

  property string pname: "Accordion Buttons"

  // thumbnailName is the file path to any Plugin logo
  // displayed in the Plugin Manager window
  // if you add a thumbnail, place it in the plugin subfolder
  // if no file is provided, a default image will be used in its place
  thumbnailName: "AccordionButtons.png"

  // title is displayed in the menu
  // and the Plugin Manager window (underneath the thumbnail)
  title: "Accordion Buttons"

  // this is displayed when you click on the icon in the plugin manager
  description: "Add accordion button symbols as lyrics"

  // categoryCode assigns the plugin to a specific sub-menu in the plugins tab
  // Currently available are:
  // "composing-arranging-tools", "color-notes", "playback", "lyrics"
  // omit for top-level menu entry
  // categoryCode: "composing-arranging-tools"

  // required for setting MS4 properties and closing Window
  id: mainWindow

  /////////////////////////////////////////////////////////////////////////
  // main entry & error handler
  /////////////////////////////////////////////////////////////////////////

  onRun: {

    if (mscoreMajorVersion < 4) {
      abortDialog.text = xsTr("The AccordionButtons plugin requires MuseScore 4 or later");
      mainWindow.parent.Window.window.close();
      abortDialog.open();
    }

    // if the version has changed, we reset all the style options to the default
    // and warn the user
    // this is necessary because the menus may have changed and the saved menu indices
    // may no longer refer to the same items that they did in the previous version
    if (dialogGrid.version != "2.3") {
      if (dialogGrid.version != "") {
 
        warningText.text = xsTr("A new version of the AccordionButtons plugin has been installed. Saved setting have been reset to the default values." );
        warningDialog.open();
 
      }
      dialogGrid.version = "2.3";
      AB.setCurrentStyleByKey('default');
      AB.putStyleToDialog();
      presetSelector.currentIndex = 0;
    }

    AB.setAnnotationOrder();
    AB.setInstrument();
    var key = AB.presetMenuOrder[presetSelector.currentIndex];
    AB.setCurrentStyleByKey(key);
    AB.configureStyleInfo();
  }

  // exception handler
  function abort(msg) {
    abortDialog.text = "AccordionButtons plugin: version 2.3\n" + msg
      + "\n" + xsTr("Please report error to") + ": Paul Anderson <paul.musescore@nine3.org>";
    abortDialog.open();
  }

  // add annotations
  // called when the OK button is pressed
  function annotate() {
    // if your plugin modifies a score, the modifications need to be enclosed by
    // curScore.startCmd() ... curScore.endCmd()
    curScore.startCmd();

    try {
      AB.annotate();
      curScore.endCmd();
    }
    catch(err) {
      curScore.endCmd();
      abort(err);
      return false;
    }

    return true;
  }

  // remove annotations
  // callled when Rest button is pressed
  function resetAnnotations() {
    curScore.startCmd();
 
   try {
      AB.resetAnnotations();
      curScore.endCmd();
    }
    catch(err) {
      curScore.endCmd();
      abort(err);
      return false;
    }
 
   return true;
  }

  function xsTr(s) { return AB.xsTr(s); }

  /////////////////////////////////////////////////////////////////////////
  // GUI - based on: https://musescore.org/en/node/345720
  /////////////////////////////////////////////////////////////////////////

  width: dialogGrid.implicitWidth
  height: dialogGrid.implicitHeight

  SystemPalette {
    id: sysActivePalette;
    colorGroup: SystemPalette.Active
  }

  SystemPalette {
    id: sysDisabledPalette;
    colorGroup: SystemPalette.Disabled
  }

/////////////////////////////////////////////////////////////////////////
// Dialog layout
/////////////////////////////////////////////////////////////////////////











/////////////////////////////////////////////////////////////////////////
// Main grid
/////////////////////////////////////////////////////////////////////////

GridLayout {

  id: dialogGrid
  columns: 2
  rowSpacing: 4

  property bool rowInside: true
  property string version: ""

  Settings { property alias rowInside: dialogGrid.rowInside }
  Settings { property alias version: dialogGrid.version }



/////////////////////////////////////////////////////////////////////////
// Instrument
/////////////////////////////////////////////////////////////////////////

  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 12
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Instrument")
  }

  RowLayout {
    ComboBox {
      id: instrumentSelector
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 12
      Layout.rightMargin: 2
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      implicitWidth: 278
      currentIndex: AB.defaultStyle.instrumentIndex
      model: AB.instrumentList()

      Settings { property alias instrument: instrumentSelector.currentIndex }

      onActivated: {
        AB.setInstrument();
      }
    }
    Button {
      id: infoButton
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 12
      Layout.rightMargin: 12
      Layout.bottomMargin: 4
      Layout.leftMargin: 0
      implicitWidth: 15
      contentItem: Text {
        text: '?'
        horizontalAlignment : Text.AlignLeft
      }
      onClicked: {
        let index = instrumentSelector.currentIndex;
        let name = AB.instrumentTables[index].name;
        let info = AB.instrumentTables[index].info;
        let author = AB.instrumentTables[index].author;
        if (info == null) info = "";
        infoDialog.title = name;
        infoText.text = (author==null)
          ? info + "<br/>"
          : info + "<br/>&nbsp;<br/><i>" + xsTr("Contributed by") + " " + author + "</i><br/>";
        infoDialog.open();
      }
    }
  }

/////////////////////////////////////////////////////////////////////////
// Variant
/////////////////////////////////////////////////////////////////////////


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Variant")
  }


  RowLayout {
    ComboBox {
      id: variantSelector
      Layout.alignment: Qt.AlignLeft
      Layout.rightMargin: 4
      Layout.leftMargin: 4
      Layout.bottomMargin: 4
      Layout.topMargin: 4
      implicitWidth: 180
      currentIndex: AB.defaultStyle.variantIndex
      model: []
      enabled: false;

      Settings { property alias instmodel: variantSelector.model }
      Settings { property alias instkey: variantSelector.currentIndex }


    }
    CheckBox {
      id: labelBox
      Layout.leftMargin: 0
      Layout.rightMargin: 4
      text: xsTr("Label")
      checked: AB.defaultStyle.showLabel

     Settings { property alias showlabel: labelBox.checked }

    }
  }

/////////////////////////////////////////////////////////////////////////
// Presets
/////////////////////////////////////////////////////////////////////////


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Preset")
  }


  RowLayout {
    ComboBox {
      id: presetSelector
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      implicitWidth: 160
      currentIndex: 0
      model: AB.presetMenuNames()

      Settings { property alias preset: presetSelector.currentIndex }

      onActivated: {
        AB.setStyleByIndex(currentIndex);
      }
    }
    Button {
      id: presetInfoButton
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 0
      Layout.rightMargin: 0
      Layout.bottomMargin: 0
      Layout.leftMargin: 0
      implicitWidth: 15
      contentItem: Text {
        text: '?'
        horizontalAlignment : Text.AlignLeft
      }
      onClicked: {
        let index = instrumentSelector.currentIndex;
        let name = presetSelector.currentValue;
        let info = AB.currentStyle.info;
        infoDialog.title = name;
        infoText.text = info + "<br/>";
        infoDialog.open();
      }
    }
    ComboBox {
      id: octaveSelector
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      implicitWidth: 107
      currentIndex: 2
      model: [ xsTr("++8ve"), xsTr("+8ve"), xsTr("Concert"), xsTr("-8ve"), xsTr("--8ve") ]

      Settings { property alias octaveKey: octaveSelector.currentIndex }


    }
  }

/////////////////////////////////////////////////////////////////////////
// Fonts
/////////////////////////////////////////////////////////////////////////


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Font")
  }


  Button {
    id: fontSelector
    Layout.alignment: Qt.AlignLeft
    Layout.topMargin: 4
    Layout.rightMargin: 4
    Layout.bottomMargin: 4
    Layout.leftMargin: 4
    implicitWidth: 300
    contentItem: Text {
        id: fontButton
        text: AB.defaultStyle.font
        horizontalAlignment : Text.AlignLeft
    }


    Settings { property alias font: fontButton.text }
    Settings { property alias fontindex: fontList.currentIndex }

    onClicked: {
      if (fontChooser.visible) {
        fontChooser.visible = false;
        fontSelector.Layout.bottomMargin = 4;
      } else {

        let f = AB.makeFontList(fontSelector.contentItem.text);
        fontList.model = f[0];
        fontList.currentIndex = f[1];
        fontChooser.visible = true;
        // when the font chooser is visible, the GridLayout is displaying
        // an additional row - so even though the dispayed rectangle has
        // zero height, an extra rowSpacing gets added to the layout
        // for all of the following items
        // removing a corresponding amount from the previous item
        // (the fontSelector) ensure that the items below this do not
        // move up and down when you open/close the selector
        fontSelector.Layout.bottomMargin = 0;
      }
    }
  }

  // see: https://www.programmersought.com/article/22736080622/
  // and: https://doc.qt.io/qt-6/qml-qtquick-listview.html
  Rectangle {
    // zero sized container so it overlays
    id: fontChooser
    visible: false
    z: 1
    width: 0
    height: 0
    Layout.columnSpan: 2
    Rectangle {
      width: { dialogGrid.width - 2*12; }
      x: 12
      height: 250
      border.color: "grey"
      color: "white"
      Component {
        id: fontListDelegate
        Item {
          id: fontListItem
          height: 20
          width: ListView.view.width
          MouseArea {
            anchors.fill: parent
            onClicked: {
              fontList.currentIndex = index;
              fontButton.text = fontList.model[index];
              fontChooser.visible = false;
            }
          }
          Text {
            text: modelData
            font.family: modelData
            font.pixelSize: 14
            leftPadding: 5
            rightPadding: 5
          }
        }
      }
      ScrollView {
        anchors.fill: parent
        ScrollBar.vertical.policy: ScrollBar.AlwaysOn
        ScrollBar.vertical.width: 10
        ListView {
          id: fontList
          anchors.fill: parent
          model: []
          delegate: fontListDelegate
          // 🤔 probably shouldn't be hard-wired - where should we get it from?
          highlight: Rectangle { color: "#69A9EB"; }
          clip: true
          highlightFollowsCurrentItem: true
        }
      }
    }
  }


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Size/Style")
  }


  RowLayout {
    ComboBox {
      id: fontSizeSelector
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      implicitWidth: 58
      currentIndex: AB.defaultStyle.fontSizeIndex
      model: [ "6", "8", "9", "10", "11", "12", "14", "16", "18", "20" ]

      Settings { property alias fontsize: fontSizeSelector.currentIndex }

    }
    ComboBox {
      id: fontStyleSelector
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      implicitWidth: 110
      currentIndex: AB.defaultStyle.fontStyleIndex
      // ⚠️ we just use the index in this array as the Musescore code for the style
      // so don't change the order!
      model: [ xsTr("Regular"), xsTr("Bold"), xsTr("Italic"), xsTr("BoldItalic") ]

      Settings { property alias fontstyle: fontStyleSelector.currentIndex }

    }
    CheckBox {
      id: padBox
      Layout.leftMargin: 0
      text: xsTr("Pad")
      checked: AB.defaultStyle.pad

     Settings { property alias pad: padBox.checked }

   }
  }

/////////////////////////////////////////////////////////////////////////
// Symbols
/////////////////////////////////////////////////////////////////////////


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Push")
  }

  RowLayout {

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: prePushSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.prePush
        maximumLength: 1
        leftPadding: 2

        Settings { property alias prePushsym: prePushSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: postChordSymInput
        KeyNavigation.tab: postPushSymInput
      }
    }

    Label {
      text: "...";
    }

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: postPushSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.postPush
        maximumLength: 1
        leftPadding: 2

        Settings { property alias postPushsym: postPushSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: prePushSymInput
        KeyNavigation.tab: prePullSymInput
      }
    }


    Button {
      id: pushOrder
      Layout.leftMargin: 12
      text: "←→"
      Layout.preferredWidth: 36
      Layout.preferredHeight: 24
      onClicked: {
        dialogGrid.rowInside = !dialogGrid.rowInside;

        AB.setAnnotationOrder();
      }
    }


    // see: https://doc.qt.io/Qt-6/qml-qtquick-controls-checkbox.html
    CheckBox {
      id: pushUnderscore
      Layout.leftMargin: 12
      Layout.rightMargin: 0
      text: "_"
      checked: AB.defaultStyle.pushUnderscore

      Settings { property alias pushunderscore: pushUnderscore.checked }

    }


    // see: https://doc.qt.io/Qt-6/qml-qtquick-controls-checkbox.html
    CheckBox {
      id: pushShift
      Layout.leftMargin: 0
      text: "↓"
      checked: AB.defaultStyle.pushShift

      Settings { property alias pushshift: pushShift.checked }

    }

  }


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Pull")
  }

  RowLayout {

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: prePullSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.prePull
        maximumLength: 1
        leftPadding: 2

        Settings { property alias prePullsym: prePullSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: postPushSymInput
        KeyNavigation.tab: postPullSymInput
      }
    }

    Label {
      text: "...";
    }

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: postPullSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.postPull
        maximumLength: 1
        leftPadding: 2

        Settings { property alias postPullsym: postPullSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: prePullSymInput
        KeyNavigation.tab: preRowSymInput
      }
    }


    Button {
      id: pullOrder
      Layout.leftMargin: 12
      text: "←→"
      Layout.preferredWidth: 36
      Layout.preferredHeight: 24
      onClicked: {
        dialogGrid.rowInside = !dialogGrid.rowInside;

        AB.setAnnotationOrder();
      }
    }


    // see: https://doc.qt.io/Qt-6/qml-qtquick-controls-checkbox.html
    CheckBox {
      id: pullUnderscore
      Layout.leftMargin: 12
      Layout.rightMargin: 0
      text: "_"
      checked: AB.defaultStyle.pullUnderscore

      Settings { property alias pullunderscore: pullUnderscore.checked }

    }


    // see: https://doc.qt.io/Qt-6/qml-qtquick-controls-checkbox.html
    CheckBox {
      id: pullShift
      Layout.leftMargin: 0
      text: "↓"
      checked: AB.defaultStyle.pullShift

      Settings { property alias pullshift: pullShift.checked }

    }

  }


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Row")
  }

  RowLayout {

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: preRowSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.preRow
        maximumLength: 1
        leftPadding: 2

        Settings { property alias preRowsym: preRowSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: postPullSymInput
        KeyNavigation.tab: postRowSymInput
      }
    }

    Label {
      text: "...";
    }

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: postRowSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.postRow
        maximumLength: 1
        leftPadding: 2

        Settings { property alias postRowsym: postRowSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: preRowSymInput
        KeyNavigation.tab: preChordSymInput
      }
    }


    Button {
      id: rowOrder
      Layout.leftMargin: 12
      text: "→←"
      Layout.preferredWidth: 36
      Layout.preferredHeight: 24
      onClicked: {
        dialogGrid.rowInside = !dialogGrid.rowInside;

        AB.setAnnotationOrder();
      }
    }

  }


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Chord")
  }

  RowLayout {

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: preChordSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.preChord
        maximumLength: 1
        leftPadding: 2

        Settings { property alias preChordsym: preChordSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: postRowSymInput
        KeyNavigation.tab: chordSeparatorSymInput
      }
    }

    Label {
      text: "...";
    }

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: chordSeparatorSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.chordSeparator
        maximumLength: 1
        leftPadding: 2

        Settings { property alias chordSeparatorsym: chordSeparatorSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: preChordSymInput
        KeyNavigation.tab: postChordSymInput
      }
    }

    Label {
      text: "...";
    }

    // see: https://programmerall.com/article/13332211499/
    Rectangle {
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 4
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      border.color: "grey"
      width: 24
      height: 24
      TextInput {
        id: postChordSymInput
        color: "black"
        anchors.fill: parent
        anchors.margins: 2
        text: AB.defaultStyle.postChord
        maximumLength: 1
        leftPadding: 2

        Settings { property alias postChordsym: postChordSymInput.text }

        Keys.onShortcutOverride: (event) => AB.overrideKey(event)
        Keys.onPressed: (event) => { text = AB.mapKey(event,text); }
        KeyNavigation.priority: KeyNavigation.BeforeItem
        KeyNavigation.backtab: chordSeparatorSymInput
        KeyNavigation.tab: prePushSymInput
      }
    }

  }

/////////////////////////////////////////////////////////////////////////
// chord style
/////////////////////////////////////////////////////////////////////////


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Chord Style")
  }


  RowLayout {
    ComboBox {
      id: chordStyleSelector
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 12
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      implicitContentWidthPolicy: ComboBox.WidestTextWhenCompleted
      currentIndex: 0
      model: [ xsTr("Default"), xsTr("Simple") ]

      Settings { property alias chordstyle: chordStyleSelector.currentIndex }

    }
  }

/////////////////////////////////////////////////////////////////////////
// alternatives
/////////////////////////////////////////////////////////////////////////


  Label {
    Layout.alignment: Qt.AlignRight
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 4
    text: xsTr("Alternatives")
  }


  RowLayout {
    ComboBox {
      id: altStyleSelector
      Layout.alignment: Qt.AlignLeft
      Layout.topMargin: 4
      Layout.rightMargin: 12
      Layout.bottomMargin: 4
      Layout.leftMargin: 4
      implicitContentWidthPolicy: ComboBox.WidestTextWhenCompleted
      currentIndex: 0
      model: [ xsTr("Not Shown"), xsTr("Regular"), xsTr("Italic"), xsTr("Grey"), xsTr("Combined") ]

     Settings { property alias altstyle: altStyleSelector.currentIndex }

    }
  }

/////////////////////////////////////////////////////////////////////////
// Dialog buttons
/////////////////////////////////////////////////////////////////////////

  DialogButtonBox {
    id: buttonBox
    Layout.columnSpan: 2
    Layout.fillWidth: true
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 4
    Layout.rightMargin: 12
    Layout.alignment: Qt.AlignHCenter
    alignment: Qt.AlignHCenter
    background: Rectangle {
        opacity: 0
    }

    // OK & Cancel buttons
    standardButtons: DialogButtonBox.Reset | DialogButtonBox.Ok | DialogButtonBox.Cancel

    // handle OK
    onAccepted: {

      if (annotate()) mainWindow.parent.Window.window.close();
    }

    // handle Reset
    onReset: {

      if (resetAnnotations()) mainWindow.parent.Window.window.close();
    }

   // handle Cancel
    onRejected: {

      mainWindow.parent.Window.window.close();
    }

  }

/////////////////////////////////////////////////////////////////////////
// Status bar
/////////////////////////////////////////////////////////////////////////

  TextArea {
    id: statusBar;
    Layout.alignment: Qt.AlignLeft
    Layout.fillWidth: true
    Layout.columnSpan: 2
    Layout.topMargin: 4
    Layout.leftMargin: 12
    Layout.bottomMargin: 12
    Layout.rightMargin: 12
    // font.pixelSize: 10
    textFormat: Text.RichText
    background: Item {}
    text: { AB.titleString(); }
    onLinkActivated: { Qt.openUrlExternally(link); }
}

}

/////////////////////////////////////////////////////////////////////////
// Info dialog
/////////////////////////////////////////////////////////////////////////

Dialog {
  id: infoDialog
  title: ""
  modal: true
  width: mainWindow.width - 2*12
  x: 12
  y: 12
  standardButtons: Dialog.Ok
  onAccepted: { infoDialog.close(); }
  TextArea {
    id: infoText
    width: infoDialog.width - 12
    wrapMode: Text.Wrap
    textFormat: Text.RichText
    background: Item {}
    padding: 10
    text: ""
    onLinkActivated: { Qt.openUrlExternally(link); }
  }
}

/////////////////////////////////////////////////////////////////////////
// Warning dialog
/////////////////////////////////////////////////////////////////////////

Dialog {
  id: warningDialog
  title: "⚠️ Warning"
  width: mainWindow.width - 2*12
  x: 12
  y: 12
  modal: true
  standardButtons: Dialog.Ok
  TextArea {
    id: warningText
    width: warningDialog.width - 12
    wrapMode: Text.Wrap
    background: Item {}
    padding: 10
    text: ""
  }
}

/////////////////////////////////////////////////////////////////////////
// Abort dialog
/////////////////////////////////////////////////////////////////////////

MessageDialog {
  id: abortDialog
  visible: false
  title: xsTr("Oops. Something went wrong.")
  onAccepted: {}
}


}
